%{
Copyright (c) 2011, MIT.
Developed with the sponsorship of the Defense Advanced Research Projects
Agency (DARPA).

Permission is hereby granted, free of charge, to any person obtaining a copy
of this data, including any software or models in source or binary form, as
well as any drawings, specifications, and documentation (collectively "the
Data"), to deal in the Data without restriction, including without
limitation the rights to use, copy, modify, merge, publish, distribute,
sublicense, and/or sell copies of the Data, and to permit persons to whom
the Data is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Data.

THE DATA IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS, SPONSORS, DEVELOPERS, CONTRIBUTORS, OR COPYRIGHT HOLDERS BE LIABLE
FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE DATA OR
THE USE OR OTHER DEALINGS IN THE DATA.
%}

% Matt Fitzgerald
% code to identify most common transition paths and design families

function [transFreq transFull implementationFamilies targets nofail notrans] = transitionAnalysis(transDesigns,plots)
% INPUT
	% transDesigns = (design,epoch) matrix of desired end states (from STRATEGY_______.m)
	% plots = 1 for display plots
% OUTPUT
	% transFreq = (design,design+2) matrix of transition frequency (last cols are steady/failure)
	% transFull = (design,design) effective transition accessibility matrix
	% targets = list of designs with which there is cause to transition to
	% nofail = list of designs with no failures in any epoch
	% notrans = list of designs that only remain steady or fail
		% note: intersection of nofail and notrans is perfectly passive value robust


totalDesigns = size(transDesigns,1);
numEpochs = size(transDesigns,2);
transFreq = zeros(totalDesigns,totalDesigns+2);
transFull = zeros(totalDesigns,totalDesigns);
targets = [];
nofail = [];
notrans = [];
Families = cell(1);

% loop through transDesigns, indexing frequency of all design--> design transitions
for d = 1:totalDesigns
	for e = 1:numEpochs
		target = transDesigns(d,e);
		if target == -1
			transFreq(d,totalDesigns+2) = transFreq(d,totalDesigns+2) + 1;
		elseif target == 0
			transFreq(d,totalDesigns+1) = transFreq(d,totalDesigns+1) + 1;
		else
			transFreq(d,target) = transFreq(d,target) + 1;
		end
	end
end

% loop through designs noting important features
for d = 1:totalDesigns
    % find all designs which might be transitioned to using this strategy
	if sum(transFreq(:,d))>0
		targets = [targets d];
    end
    % find all designs which always are acceptable or have an acceptable
    % transition
	if transFreq(d,totalDesigns+2) == 0
		nofail = [nofail d];
    end
    % find all designs which always either stay or fail (no transitions)
	if transFreq(d,totalDesigns+1) + transFreq(d,totalDesigns+2) == numEpochs
		notrans = [notrans d];
	end
end

% iterate until effective transition accessibility is found
transFull = transFreq(1:totalDesigns,1:totalDesigns)>0;
PREVtransFull = zeros(totalDesigns,totalDesigns);
iter = 0;
while ~isequal(PREVtransFull,transFull)
    iter = iter+1
	PREVtransFull = transFull;
	for d = 1:totalDesigns
		for d2 = 1:totalDesigns
			if transFull(d,d2) == 1  % if d can transition to d2
				addTargets = transFull(d2,:)==1;   % find all the designs d2 can transition to
				transFull(d,:) = transFull(d,:) + addTargets; % add an index to the d row for each of those end states
			end
		end
		transFull(d,:) = transFull(d,:)>0; % reset all positive entries in the d row to one
	end
end



% this section finds "implementationFamilies" = families of design numbers
% which, using the given strategy, result in infinite loops between them
    % NOTE: not all systems and strategies will result in looping families,
    % particularly if there are designs availiable that fit into the
    % "notrans" category, as these designs act like network sinks
circularDesigns = [];
norepeats = cell(1);
% identify circularDesigns = designs which can transition away and back
for d = 1:totalDesigns
    if transFull(d,d) == 1
        circularDesigns = [circularDesigns d];
    end
end
% potential families (Families) are the implemented accessibilities of all
% the circular designs
for cd = circularDesigns
        family = find(transFull(cd,:)>0);
        Families = [Families family];
end
% eliminate repeats (a family of 3 members will have up to 3 repeats)
for f = 1:length(Families)
    for i = 1:length(norepeats)
        if isequal(Families{f},norepeats{i})
            break
        end
        if i == length(norepeats)
            norepeats = [norepeats Families(f)];
        end
    end
end
norepeats = norepeats(2:end);
% eliminate families that are supersets of smaller families, as once you
% transition into the small family, you cannot leave again
for i = 1:length(norepeats)
    currFam = norepeats{i};
    for j = 1:length(norepeats)
        if i ~= j && isequal(intersect(currFam,norepeats{j}),currFam)
            norepeats{j} = [];
        end
    end
end
norepeats(cellfun(@isempty,norepeats)) = [];
% the result is the implementationFamilies
implementationFamilies = norepeats;




%% PLOTS
if plots
	% spy plot the effective accessibility
	figure()
	spy(transFull)
	title('Spyplot of Effective Accessibility with this Strategy')
	xlabel('Design End States')
	ylabel('Design Begin States')
	
	% plot the # transitions to each designs
	figure()
	plot(1:totalDesigns,sum((transFreq(:,1:totalDesigns))))
	title('Transition End State Frequency')
    xlabel('Design End States')
    ylabel('Frequency')
    ylim([0 max(sum(transFreq(1:totalDesigns,1:totalDesigns)))+1])
    
    % plot an example era network
    initialDesign = 65;
    possibleStates{1} = initialDesign;
    for i = 1:50
        possibleStates{i+1} = [];
        for j = 1:length(possibleStates{i})
            [x newStates] = find(transFreq(possibleStates{i}(j),:)>0);
            possibleStates{i+1} = [possibleStates{i+1} newStates];
            
            possibleStates{i+1} = sort(possibleStates{i+1});
            
            if intersect(totalDesigns+2,possibleStates{i+1})==totalDesigns+2
                possibleStates{i+1}(end) = [];
            end
            if intersect(totalDesigns+1,possibleStates{i+1})==totalDesigns+1
                possibleStates{i+1}(end) = [];
                possibleStates{i+1} = [possibleStates{i}(j) possibleStates{i+1}];
            end
        end
        possibleStates{i+1} = unique(possibleStates{i+1});
        if isequal(possibleStates{i+1},possibleStates{i})
            break
        end
    end
    figure()
    hold on
    for i = 1:length(possibleStates)-1
        for j = 1:length(possibleStates{i})
            [x goto] = find(transFreq(possibleStates{i}(j),:)>0);
            if intersect(totalDesigns+2,goto)==totalDesigns+2
                goto(end) = [];
                goto = [-100 goto];
            end
            if intersect(totalDesigns+1,goto)==totalDesigns+1
                goto(end) = [];
                goto = [possibleStates{i}(j) goto];
            end
            for k = goto
                plot([i,i+1],[possibleStates{i}(j),k])
            end
        end
    end
    title(['Era Network for Design _',num2str(initialDesign),'_'])
    ylabel('Design Number')
    xlabel('Era epochs')
    
end


end
